#!/usr/bin/env python
# -*- coding: utf_8 -*-
import logging
import traceback
from flask import Blueprint, request
from app import config, signalManager
from fullstack.login import Auth
from fullstack.validation import validate_schema
from fullstack.response import ResponseCode, response_result
from schema.user import AddSchema, DeleteSchema, UpdateSchema, QuerySchema

logger = logging.getLogger(__name__)

user_api = Blueprint("user_api", __name__, url_prefix="/api/v1/%s/user" % config['NAME'])

@user_api.route("/add", methods=['POST'])
@validate_schema(AddSchema)
@Auth.auth_required
def add():
    '''
    新增用户
    '''
    try:
        isSuccess, message = signalManager.actionAddUser.emit(request.current_user.get("id"), request.schema_data)
        if isSuccess:
            return response_result(ResponseCode.OK, msg=message)
        else:
            return response_result(ResponseCode.REQUEST_ERROR, msg=message)
    except Exception as e:
        traceback.print_exc()
        return response_result(ResponseCode.SERVER_ERROR, msg=str(e))


@user_api.route('/delete/<uuid:id>', methods=['POST'])
@validate_schema(DeleteSchema)
@Auth.auth_required
def delete(id):
    '''
    删除用户
    '''
    try:
        isSuccess, message = signalManager.actionDeleteUser.emit(id)
        if isSuccess:
            return response_result(ResponseCode.OK, msg=message)
        else:
            return response_result(ResponseCode.REQUEST_ERROR, msg=message)
    except Exception as e:
        traceback.print_exc()
        logger.error(str(e))
        return response_result(ResponseCode.SERVER_ERROR)


@user_api.route('/get', methods=['POST'])
@Auth.auth_required
def get():
    '''
    查找单个用户
    '''
    try:
        result, message = signalManager.actionGetUser.emit(request.current_user.get("id"))
        if result:
            return response_result(ResponseCode.OK, data=result, msg=message)
        else:
            if result == None:
                return response_result(ResponseCode.NO_DATA_FOUND)
            return response_result(ResponseCode.REQUEST_ERROR, msg=message)
    except Exception as e:
        traceback.print_exc()
        logger.error(str(e))
        return response_result(ResponseCode.SERVER_ERROR, msg=str(e))


@user_api.route('/list', methods=['POST'])
@validate_schema(QuerySchema)
# @Auth.auth_required
def getList():
    '''
    查询用户列表
    '''
    try:
        result, count, message = signalManager.actionGetUserList.emit(request.schema_data)
        if result:
            return response_result(ResponseCode.OK, data=result, msg=message, count=count)
        else:
            return response_result(ResponseCode.REQUEST_ERROR, msg=message)
    except Exception as e:
        traceback.print_exc()
        logger.error(str(e))
        return response_result(ResponseCode.SERVER_ERROR)

@user_api.route('/update/<uuid:id>', methods=['POST'])
@validate_schema(UpdateSchema)
# @Auth.auth_required
def update(id):
    '''
    更新用户
    '''
    try:
        result, message = signalManager.actionUpdateUser.emit(id, request.schema_data)
        if result:
            return response_result(ResponseCode.OK, data=result, msg=message)
        else:
            return response_result(ResponseCode.NOTHING_CHANGE, msg=message)
    except Exception as e:
        traceback.print_exc()
        logger.error(str(e))
        return response_result(ResponseCode.SERVER_ERROR)
